import datetime
from collections.abc import Mapping
from typing import Any, TypeVar, Union
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..models.history_revision_object_action import HistoryRevisionObjectAction
from ..models.history_revision_request_action import HistoryRevisionRequestAction
from ..types import UNSET, Unset

T = TypeVar("T", bound="HistoryRevision")


@_attrs_define
class HistoryRevision:
    """
    Attributes:
        id (UUID):  Example: b19d819a-5baf-46ae-88b6-7e6f6f0c10d7.
        object_name (str):
        timestamp (datetime.datetime):
        request_action (HistoryRevisionRequestAction):
        object_path (str):
        object_action (HistoryRevisionObjectAction):
        request_path (str):
        user (str):
        payload (str):
        object_id (Union[Unset, UUID]):
        revision (Union[Unset, UUID]):
    """

    id: UUID
    object_name: str
    timestamp: datetime.datetime
    request_action: HistoryRevisionRequestAction
    object_path: str
    object_action: HistoryRevisionObjectAction
    request_path: str
    user: str
    payload: str
    object_id: Union[Unset, UUID] = UNSET
    revision: Union[Unset, UUID] = UNSET
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        id = str(self.id)

        object_name = self.object_name

        timestamp = self.timestamp.isoformat()

        request_action = self.request_action.value

        object_path = self.object_path

        object_action = self.object_action.value

        request_path = self.request_path

        user = self.user

        payload = self.payload

        object_id: Union[Unset, str] = UNSET
        if not isinstance(self.object_id, Unset):
            object_id = str(self.object_id)

        revision: Union[Unset, str] = UNSET
        if not isinstance(self.revision, Unset):
            revision = str(self.revision)

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "id": id,
                "objectName": object_name,
                "timestamp": timestamp,
                "requestAction": request_action,
                "objectPath": object_path,
                "objectAction": object_action,
                "requestPath": request_path,
                "user": user,
                "payload": payload,
            }
        )
        if object_id is not UNSET:
            field_dict["objectId"] = object_id
        if revision is not UNSET:
            field_dict["revision"] = revision

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        d = dict(src_dict)
        id = UUID(d.pop("id"))

        object_name = d.pop("objectName")

        timestamp = isoparse(d.pop("timestamp"))

        request_action = HistoryRevisionRequestAction(d.pop("requestAction"))

        object_path = d.pop("objectPath")

        object_action = HistoryRevisionObjectAction(d.pop("objectAction"))

        request_path = d.pop("requestPath")

        user = d.pop("user")

        payload = d.pop("payload")

        _object_id = d.pop("objectId", UNSET)
        object_id: Union[Unset, UUID]
        if isinstance(_object_id, Unset):
            object_id = UNSET
        else:
            object_id = UUID(_object_id)

        _revision = d.pop("revision", UNSET)
        revision: Union[Unset, UUID]
        if isinstance(_revision, Unset):
            revision = UNSET
        else:
            revision = UUID(_revision)

        history_revision = cls(
            id=id,
            object_name=object_name,
            timestamp=timestamp,
            request_action=request_action,
            object_path=object_path,
            object_action=object_action,
            request_path=request_path,
            user=user,
            payload=payload,
            object_id=object_id,
            revision=revision,
        )

        history_revision.additional_properties = d
        return history_revision

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
