import datetime
from collections.abc import Mapping
from typing import Any, TypeVar, Union
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..models.file_status import FileStatus
from ..models.quarantine_reason import QuarantineReason
from ..types import UNSET, Unset

T = TypeVar("T", bound="FileDescription")


@_attrs_define
class FileDescription:
    """
    Attributes:
        id (UUID):  Example: 158fb228-d101-448f-854d-2101de4d09a6.
        file_name (str):
        mime_type (str):
        created_by (str):
        status (FileStatus):
        uploaded_date (Union[Unset, datetime.datetime]): Upload Timestamp Example: 2000-10-31 01:30:00-05:00.
        quarantine_reason (Union[Unset, QuarantineReason]):
        checking_progress_in_percent (Union[Unset, int]):
    """

    id: UUID
    file_name: str
    mime_type: str
    created_by: str
    status: FileStatus
    uploaded_date: Union[Unset, datetime.datetime] = UNSET
    quarantine_reason: Union[Unset, QuarantineReason] = UNSET
    checking_progress_in_percent: Union[Unset, int] = UNSET
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        id = str(self.id)

        file_name = self.file_name

        mime_type = self.mime_type

        created_by = self.created_by

        status = self.status.value

        uploaded_date: Union[Unset, str] = UNSET
        if not isinstance(self.uploaded_date, Unset):
            uploaded_date = self.uploaded_date.isoformat()

        quarantine_reason: Union[Unset, str] = UNSET
        if not isinstance(self.quarantine_reason, Unset):
            quarantine_reason = self.quarantine_reason.value

        checking_progress_in_percent = self.checking_progress_in_percent

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "id": id,
                "fileName": file_name,
                "mimeType": mime_type,
                "createdBy": created_by,
                "status": status,
            }
        )
        if uploaded_date is not UNSET:
            field_dict["uploadedDate"] = uploaded_date
        if quarantine_reason is not UNSET:
            field_dict["quarantineReason"] = quarantine_reason
        if checking_progress_in_percent is not UNSET:
            field_dict["checkingProgressInPercent"] = checking_progress_in_percent

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        d = dict(src_dict)
        id = UUID(d.pop("id"))

        file_name = d.pop("fileName")

        mime_type = d.pop("mimeType")

        created_by = d.pop("createdBy")

        status = FileStatus(d.pop("status"))

        _uploaded_date = d.pop("uploadedDate", UNSET)
        uploaded_date: Union[Unset, datetime.datetime]
        if isinstance(_uploaded_date, Unset):
            uploaded_date = UNSET
        else:
            uploaded_date = isoparse(_uploaded_date)

        _quarantine_reason = d.pop("quarantineReason", UNSET)
        quarantine_reason: Union[Unset, QuarantineReason]
        if isinstance(_quarantine_reason, Unset):
            quarantine_reason = UNSET
        else:
            quarantine_reason = QuarantineReason(_quarantine_reason)

        checking_progress_in_percent = d.pop("checkingProgressInPercent", UNSET)

        file_description = cls(
            id=id,
            file_name=file_name,
            mime_type=mime_type,
            created_by=created_by,
            status=status,
            uploaded_date=uploaded_date,
            quarantine_reason=quarantine_reason,
            checking_progress_in_percent=checking_progress_in_percent,
        )

        file_description.additional_properties = d
        return file_description

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
