import datetime
from collections.abc import Mapping
from typing import Any, TypeVar, Union
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..models.abstract_consent_invitation_consent_scope import AbstractConsentInvitationConsentScope
from ..models.consent_permission import ConsentPermission
from ..models.invitation_status import InvitationStatus
from ..models.invitation_type import InvitationType
from ..types import UNSET, Unset

T = TypeVar("T", bound="ConsentInvitation")


@_attrs_define
class ConsentInvitation:
    """
    Attributes:
        invitation_type (InvitationType):  Example: ADD.
        valid_until_date (datetime.date):  Example: 2028-05-30.
        id (UUID):  Example: de7f0e04-74fb-4657-9039-03dee0e6049b.
        invitation_status (InvitationStatus):  Example: ACTIVE.
        token (str):
        consent_scope (AbstractConsentInvitationConsentScope):  Example: USER_CONTEXT.
        consent_permission (ConsentPermission):  Example: READ.
        consent_valid_until_date (Union[Unset, datetime.datetime]):  Example: 2029-05-30T00:00:00.
    """

    invitation_type: InvitationType
    valid_until_date: datetime.date
    id: UUID
    invitation_status: InvitationStatus
    token: str
    consent_scope: AbstractConsentInvitationConsentScope
    consent_permission: ConsentPermission
    consent_valid_until_date: Union[Unset, datetime.datetime] = UNSET
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        invitation_type = self.invitation_type.value

        valid_until_date = self.valid_until_date.isoformat()

        id = str(self.id)

        invitation_status = self.invitation_status.value

        token = self.token

        consent_scope = self.consent_scope.value

        consent_permission = self.consent_permission.value

        consent_valid_until_date: Union[Unset, str] = UNSET
        if not isinstance(self.consent_valid_until_date, Unset):
            consent_valid_until_date = self.consent_valid_until_date.isoformat()

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "invitationType": invitation_type,
                "validUntilDate": valid_until_date,
                "id": id,
                "invitationStatus": invitation_status,
                "token": token,
                "consentScope": consent_scope,
                "consentPermission": consent_permission,
            }
        )
        if consent_valid_until_date is not UNSET:
            field_dict["consentValidUntilDate"] = consent_valid_until_date

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        d = dict(src_dict)
        invitation_type = InvitationType(d.pop("invitationType"))

        valid_until_date = isoparse(d.pop("validUntilDate")).date()

        id = UUID(d.pop("id"))

        invitation_status = InvitationStatus(d.pop("invitationStatus"))

        token = d.pop("token")

        consent_scope = AbstractConsentInvitationConsentScope(d.pop("consentScope"))

        consent_permission = ConsentPermission(d.pop("consentPermission"))

        _consent_valid_until_date = d.pop("consentValidUntilDate", UNSET)
        consent_valid_until_date: Union[Unset, datetime.datetime]
        if isinstance(_consent_valid_until_date, Unset):
            consent_valid_until_date = UNSET
        else:
            consent_valid_until_date = isoparse(_consent_valid_until_date)

        consent_invitation = cls(
            invitation_type=invitation_type,
            valid_until_date=valid_until_date,
            id=id,
            invitation_status=invitation_status,
            token=token,
            consent_scope=consent_scope,
            consent_permission=consent_permission,
            consent_valid_until_date=consent_valid_until_date,
        )

        consent_invitation.additional_properties = d
        return consent_invitation

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
