from http import HTTPStatus
from typing import Any, Optional, Union, cast
from uuid import UUID

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.file_get_description import FileGetDescription
from ...types import Response


def _get_kwargs(
    user_context: UUID,
    file_id: UUID,
) -> dict[str, Any]:
    _kwargs: dict[str, Any] = {
        "method": "get",
        "url": f"/file-manager/v1/{user_context}/files/{file_id}",
    }

    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, FileGetDescription]]:
    if response.status_code == 200:
        response_200 = FileGetDescription.from_dict(response.json())

        return response_200
    if response.status_code == 400:
        response_400 = cast(Any, None)
        return response_400
    if response.status_code == 401:
        response_401 = cast(Any, None)
        return response_401
    if response.status_code == 403:
        response_403 = cast(Any, None)
        return response_403
    if response.status_code == 404:
        response_404 = cast(Any, None)
        return response_404
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, FileGetDescription]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    user_context: UUID,
    file_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Response[Union[Any, FileGetDescription]]:
    """Get the Attributes of the file

     Return the File Description.

    Args:
        user_context (UUID):  Example: 7df13462-54d8-4fd7-a2e8-5e18754d4030.
        file_id (UUID):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, FileGetDescription]]
    """

    kwargs = _get_kwargs(
        user_context=user_context,
        file_id=file_id,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    user_context: UUID,
    file_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Optional[Union[Any, FileGetDescription]]:
    """Get the Attributes of the file

     Return the File Description.

    Args:
        user_context (UUID):  Example: 7df13462-54d8-4fd7-a2e8-5e18754d4030.
        file_id (UUID):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, FileGetDescription]
    """

    return sync_detailed(
        user_context=user_context,
        file_id=file_id,
        client=client,
    ).parsed


async def asyncio_detailed(
    user_context: UUID,
    file_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Response[Union[Any, FileGetDescription]]:
    """Get the Attributes of the file

     Return the File Description.

    Args:
        user_context (UUID):  Example: 7df13462-54d8-4fd7-a2e8-5e18754d4030.
        file_id (UUID):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, FileGetDescription]]
    """

    kwargs = _get_kwargs(
        user_context=user_context,
        file_id=file_id,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    user_context: UUID,
    file_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Optional[Union[Any, FileGetDescription]]:
    """Get the Attributes of the file

     Return the File Description.

    Args:
        user_context (UUID):  Example: 7df13462-54d8-4fd7-a2e8-5e18754d4030.
        file_id (UUID):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, FileGetDescription]
    """

    return (
        await asyncio_detailed(
            user_context=user_context,
            file_id=file_id,
            client=client,
        )
    ).parsed
